#!/usr/bin/env python3

import sys, grpc, re
from time import sleep

from nuance.nrc.v1.nrc_pb2 import *
from nuance.nrc.v1.nrc_pb2_grpc import *


def usage():
    print (f'''
Usage:
    {sys.argv[0]} <server_address> <access_token> <audio_file>

    server_address - address of the server with a port in the form server:port
      access_token - Mix access token string
        audio_file - file containing audio to recognize, ex. "audio/01234.ulaw"

Examples:
    $ export TOKEN=<access_token>
    $ {sys.argv[0]} nr.api.nuance.com:443 $TOKEN audio/01234.ulaw    // will recognize a "0-1-2-3-4" audio utterance.
    $ {sys.argv[0]} nr.api.nuance.com:443 $TOKEN audio/orange.ulaw   // will recognize the word "orange".

    // Note: By default the client expects a raw mu-law encoded audio file.
'''
    )

#----------------------------------------------------------------------------------------------------
# Builtin grammar
#----------------------------------------------------------------------------------------------------
builtin_digits_grammar = RecognitionResource(
    builtin = "builtin:grammar/digits",
    language="en-US",
    weight=1
)

#----------------------------------------------------------------------------------------------------
# Inline grammar
#----------------------------------------------------------------------------------------------------
inline_grammar_str = '''<?xml version="1.0" encoding="UTF-8"?>
<grammar xmlns="http://www.w3.org/2001/06/grammar" xml:lang="en-US" version="1.0" root="colors">
    <rule id="colors" scope="public">
        <one-of>
            <item>red</item>
            <item>blue</item>
            <item>green</item>
            <item>yellow</item>
            <item>orange</item>
            <item>black</item>
            <item>white</item>
        </one-of>
    </rule>
</grammar>
'''

# The language should correspond to the language of the grammar.
inline_colors_grammar = RecognitionResource(
    inline_grammar = InlineGrammar(
        # For a more compact string, remove newlines and spaces between xml tags.
        grammar = bytes(re.sub('>\s*', '>', inline_grammar_str), 'utf-8'),
        media_type=1
    ),
    language="en-US",
    weight=1
)

#----------------------------------------------------------------------------------------------------
# URI grammar
#----------------------------------------------------------------------------------------------------
# This example uri grammar points to a file on a http server.
# The language should correspond to the language of the grammar.
uri_fruits_grammar = RecognitionResource(
    uri_grammar = UriGrammar(
        uri="http://myserver/mygrammars/fruits.grxml",
        media_type=1
    ),
    language="en-US",
    weight = 1
)


def recognition_init():
    init = RecognitionInit(
        parameters = RecognitionParameters(
            audio_format = AudioFormat(ulaw = ULaw()),  # default, other options are ALaw and PCM.
            no_input_timeout_ms = 2000,                 # default is 7000 milliseconds.
            confidence_level = 0.4                      # return a no-match if the confidence is less than 0.4.
        ),
        resources = [
            builtin_digits_grammar,
            inline_colors_grammar,
            #uri_fruits_grammar
        ]
    )
    return init


def client_stream(audio_file):
    # Start the recognition
    init = recognition_init()

    # Log init message.
    initstr = re.sub('^|\n', '\n  ', repr(init)) # indent init message for better readability
    initstr = re.sub('\s*$', '', initstr)        # remove extra spaces at end of string
    print("Sending recognition_init {" + initstr + "\n}\n")

    yield RecognitionRequest(recognition_init = init)

    # Send audio in packets of 160 bytes (for mu-law: 1 byte = 1 audio sample).
    # 160 bytes of mu-law audio at 8kHz = 20 milliseconds of audio
    packet_size = 160
    packet_duration = 0.020

    packet = 0 # To log how many packets are sent.
    try :
        with open(audio_file, "rb") as f:
            while True:
                data = f.read(packet_size)
                if not data:
                    break
                else:
                    packet += 1
                    print ("Sending audio packet " + repr(packet) + " length " + repr(len(data)))
                    yield RecognitionRequest(audio = bytes(data))
                    # Simulate audio streaming by sleeping for the duration of the audio packet.
                    # Note that this is just for this demo, it is not required.
                    sleep(packet_duration)
    except Exception as e:
        print("File read exception" + repr(e))

    print("DONE Sending audio");


def recognize(server_address, access_token, audio_file):
    call_credentials = grpc.access_token_call_credentials(access_token)
    ssl_credentials = grpc.ssl_channel_credentials()
    channel_credentials = grpc.composite_channel_credentials(ssl_credentials, call_credentials)

    with grpc.secure_channel(server_address, credentials=channel_credentials) as channel:

        nrc = NRCStub(channel)

        response_iterator = nrc.Recognize(client_stream(audio_file))
        try:
            for response in response_iterator:
                print("\nRecognize() reponse --> " + repr(response))
        except Exception as e:
            print('Server stream exception: ' + repr(e))


if __name__ == '__main__':
    server_address = access_token = audio_file = None
    try:
        server_address = sys.argv[1]
        access_token = sys.argv[2]
        audio_file = sys.argv[3]
    except Exception as e:
        usage()
        exit(1)
    recognize(server_address, access_token, audio_file)
