#!/usr/bin/env python3

import sys, grpc, re
from time import sleep

from nuance.nrc.v1.nrc_pb2 import *
from nuance.nrc.v1.nrc_pb2_grpc import *


def usage():
    print (f'''
Usage:
    {sys.argv[0]} <server_address> <access_token> <dtmfsequence>

    server_address - address of the server with a port in the form server:port
      access_token - Mix access token string
     dtmf_sequence - string of dtmf digits to recognize, ex. "1234#"
                     You can also send a preset dtmf sequence of the first 7
                     digits of these 4 numbers: pi, e, phi, sqrt2

Examples:
    $ export TOKEN=<access_token>
    $ {sys.argv[0]} nr.api.nuance.com:443 $TOKEN "1234"        // will send the sequence 1234
    $ {sys.argv[0]} nr.api.nuance.com:443 $TOKEN "3*14159"     // will send the sequence 3*14159
    $ {sys.argv[0]} nr.api.nuance.com:443 $TOKEN "pi"          // will send the sequence 3*14159
    $ {sys.argv[0]} nr.api.nuance.com:443 $TOKEN "sqrt2"       // will send the sequence 1*41421
    $ {sys.argv[0]} nr.api.nuance.com:443 $TOKEN "1234#1234"   // will send the sequence 1234#1234

    // Note: The 1234#1324 recognition will end on the # because dtmf_term_char is set to #.
'''
    )

#----------------------------------------------------------------------------------------------------
# Builtin DTMF grammar
#----------------------------------------------------------------------------------------------------
builtin_dtmf_digits_grammar = RecognitionResource(
    builtin = "builtin:dtmf/digits",
    weight = 1
)

#----------------------------------------------------------------------------------------------------
# Inline DTMF grammar
#----------------------------------------------------------------------------------------------------
inline_dtmf_numbers_grammar_str = '''<?xml version="1.0" encoding="UTF-8"?>
<grammar xml:lang="en-US" mode="dtmf" version="1.0" tag-format="swi-semantics/1.0" root="NUMBERS" xmlns="http://www.w3.org/2001/06/grammar">
    <rule id="NUMBERS" scope="public">
        <one-of>
            <item><ruleref uri="#PI" />    <tag>MEANING=PI.SWI_literal.replace(/.*/, 'PI')</tag></item>
            <item><ruleref uri="#E" />     <tag>MEANING=E.SWI_literal.replace(/.*/, 'E')</tag></item>
            <item><ruleref uri="#PHI" />   <tag>MEANING=PHI.SWI_literal.replace(/.*/, 'PHI')</tag></item>
            <item><ruleref uri="#SQRT2" /> <tag>MEANING=SQRT2.SWI_literal.replace(/.*/, 'SQRT2')</tag></item>
        </one-of>
    </rule>
    <rule id="PI">    <item>3</item><item>*</item><item>1</item><item>4</item><item>1</item><item>5</item><item>9</item></rule>
    <rule id="E">     <item>2</item><item>*</item><item>7</item><item>1</item><item>8</item><item>2</item><item>8</item></rule>
    <rule id="PHI">   <item>1</item><item>*</item><item>6</item><item>1</item><item>8</item><item>0</item><item>3</item></rule>
    <rule id="SQRT2"> <item>1</item><item>*</item><item>4</item><item>1</item><item>4</item><item>2</item><item>1</item></rule>
</grammar>
'''

inline_dtmf_numbers_grammar = RecognitionResource(
    inline_grammar = InlineGrammar(
        # For a more compact string, remove newlines and spaces between xml tags.
        grammar = bytes(re.sub('>\s*', '>', inline_dtmf_numbers_grammar_str), 'utf-8')
    ),
    weight = 2  # give a preference to this grammar over the digits grammar.
)

#----------------------------------------------------------------------------------------------------
# URI DTMF grammar
#----------------------------------------------------------------------------------------------------
# This example uri DTMF grammar points to a file on a http server.
uri_dtmf_months_grammar = RecognitionResource(
    uri_grammar = UriGrammar(
        uri="http://myserver/mygrammars/dtmf_months.grxml"
    ),
    weight = 1
)


def dtmf_recognition_init():
    init = DTMFRecognitionInit(
        parameters = DTMFRecognitionParameters(
            no_input_timeout_ms = 6000,         # default is 7000 milliseconds
            dtmf_interdigit_timeout_ms = 3000,  # default is 5000 milliseconds
            dtmf_term_char = "#"                # signal the end of input with a DTMF #
        ),
        resources = [
            builtin_dtmf_digits_grammar,
            inline_dtmf_numbers_grammar,
            #uri_dtmf_months_grammar,
        ]
    )
    return init


def dtmf_stream(dtmf_sequence):
    # Start the recognition
    init = dtmf_recognition_init()

    # Log init message.
    initstr = re.sub('^|\n', '\n  ', repr(init)) # indent init message for better readability
    initstr = re.sub('\s*$', '', initstr)        # remove extra spaces at end of string
    print("Sending recognition_init {" + initstr + "\n}\n")

    yield DTMFRecognitionRequest(recognition_init = init)

    # Sleep 1 second to simulate some delay between the start of the DTMF
    # recognition request and the first incoming DTMF digit.
    # Note that this is just for this demo, it is not required.
    sleep(1.0)

    # Send DTMF digits on the stream, 1 at a time, with 0.1 second delay between digits.
    # NOTE: digits can be sent in a single string, they don't need to be sent individually.
    #       example: yield DTMFRecognitionRequest(dtmf = "1234#")
    for digit in dtmf_sequence:
        print("Sending DTMF " + digit);
        dtmfstr = "" + digit
        yield DTMFRecognitionRequest(dtmf = dtmfstr)
        # Sleep 0.1 second to simulate some delay between DTMF digits.
        # Note that this is just for this demo, it is not required.
        sleep(0.1)

    print("DONE Sending DTMFs");


def recognize(server_address, access_token, dtmf_sequence_str):

    call_credentials = grpc.access_token_call_credentials(access_token)
    ssl_credentials = grpc.ssl_channel_credentials()
    channel_credentials = grpc.composite_channel_credentials(ssl_credentials, call_credentials)

    with grpc.secure_channel(server_address, credentials=channel_credentials) as channel:

        nrc = NRCStub(channel)

        # Select which sequence of dtmf to send.
        # By default select the sequence specified on the command line.
        dtmf_sequence_str = dtmf_sequence_str.lower()
        if   (dtmf_sequence_str == 'pi'):    sequence = '3*14159'
        elif (dtmf_sequence_str == 'e'):     sequence = '2*71828'
        elif (dtmf_sequence_str == 'phi'):   sequence = '1*61803'
        elif (dtmf_sequence_str == 'sqrt2'): sequence = '1*41421'
        else:                                sequence =  dtmf_sequence_str

        dtmf_sequence = list(sequence)
        print("Recognizing DTMF sequence: " + repr(dtmf_sequence) + "\n")
        response_iterator = nrc.DTMFRecognize(dtmf_stream(dtmf_sequence))
        try:
            for response in response_iterator:
                print("\nDTMFRecognize() reponse --> " + repr(response))
        except Exception as e:
            print('Server stream exception: ' + repr(e))


if __name__ == '__main__':
    server_address = access_token = dtmf_sequence = None
    try:
        server_address = sys.argv[1]
        access_token = sys.argv[2]
        dtmf_sequence = sys.argv[3]
    except Exception as e:
        usage()
        exit(1)
    recognize(server_address, access_token, dtmf_sequence)
