import argparse
import time
import logging
import grpc
import urllib
import urllib.request
import base64
import json
import threading

from google.protobuf import text_format
from google.protobuf import json_format

from google.protobuf.json_format import MessageToJson

from nuance.nlu.wordset.v1beta1.wordset_pb2 import *
from nuance.nlu.wordset.v1beta1.wordset_pb2_grpc import *
from nuance.nlu.common.v1beta1.resource_pb2 import *

from google.protobuf.json_format import MessageToJson

oauth_token_expiry_threshhold_seconds = 30
oauth_token_expiry_seconds = 0
oauth_token = None

args = None

def parse_args():
    global args
    parser = argparse.ArgumentParser(
        prog="nlu_wordset_client.py",
        usage="%(prog)s <command> [-options]",
        add_help=False,
        formatter_class=lambda prog: argparse.HelpFormatter(
            prog, max_help_position=45, width=100)
    )

    parser.add_argument('command', metavar='command', nargs='?',
        choices=['compile', 'get-metadata', 'delete'], 
        help="Command to execute [values: compile, get-metadata, delete] (default: compile)", default='compile')
    options = parser.add_argument_group("options")
    options.add_argument("-h", "--help", action="help",
                         help="Show this help message and exit.")
    options.add_argument("--token", nargs="?", help=argparse.SUPPRESS)
    options.add_argument("--oauthURL", metavar="oauthUrl", nargs="?",
                         help="OAuth 2.0 URL")
    options.add_argument("--clientID", metavar="clientID", nargs="?",
                         help="OAuth 2.0 Client ID")
    options.add_argument("--clientSecret", metavar="clientSecret", nargs="?",
                         help="OAuth 2.0 Client Secret")
    options.add_argument("--oauthScope", metavar="oauthScope", nargs="?",
                         help="OAuth 2.0 Scope, default=nlu nlu.wordset", default='nlu nlu.wordset')
    options.add_argument("-s", "--serverUrl", metavar="serverUrl", nargs="?",
                         help="NLU server URL, default=localhost:9090.", default='localhost:9090')
    options.add_argument("--wordsetFile", type=argparse.FileType("r"), metavar="file",
                         nargs="?", help="Wordset JSON file.")
    options.add_argument("--artifactUrn", nargs="?", metavar="artifactUrn", help="Compiled Wordset URN.")
    options.add_argument("--modelUrn", nargs="?", metavar="modelUrn", help="NLU Model URN.")
    options.add_argument("--metadata", metavar="metadata", nargs="?",
                         help="Wordset metadata defined as one or more key:value pairs.", default=[],)
    options.add_argument("--clientData", metavar="clientData", nargs="?", 
                         help="Client data defined as one or more key=value pairs.", default=[])
    args = parser.parse_args()

def get_oauth2_token():
    global oauth_token
    global oauth_token_expiry_seconds
    global oauth_token_expiry_threshhold_seconds

    if args.oauthURL is None:
        return None
    
    current_time = time.monotonic()

    try:
        if oauth_token and oauth_token_expiry_seconds - oauth_token_expiry_threshhold_seconds > current_time:
            log.debug('OAuth token is still valid')
            return oauth_token

        log.info("Obtaining auth token (Client ID: {}, URL: {})".format(args.clientID, args.oauthURL))

        encoded_credentials = base64.standard_b64encode("{}:{}".format(args.clientID, args.clientSecret).encode()).decode('utf-8')
        headers = { 'Authorization' : "Basic {}".format(encoded_credentials)  }

        data = {
            'grant_type': 'client_credentials',
            'scope': args.oauthScope,
        }

        request = urllib.request.Request(url=args.oauthURL, headers=headers, data=urllib.parse.urlencode(data).encode(), method='POST')
        with urllib.request.urlopen(request) as response:
            response = response.read().decode('utf-8')
            json_response = json.loads(response)

            oauth_token = json_response["access_token"]
            oauth_token_expiry_seconds = time.monotonic() + json_response["expires_in"]
        
            log.debug("Token TTL: %d" % json_response["expires_in"])
            return json_response["access_token"]
    except urllib.error.HTTPError as err:
        raise Exception("Failed to obtain authentication token. Status: {}, Error: {}".format(err.code, err.read().decode()))

def create_channel(args):    
    call_credentials = None
    channel = None

    if args.token:
        log.debug('Adding CallCredentials using token parameter')
        call_credentials = grpc.access_token_call_credentials(args.token)
    else:
        current_oauth_token = get_oauth2_token()
        if current_oauth_token:
            log.debug('Adding CallCredentials from OAuth endpoint')
            call_credentials = grpc.access_token_call_credentials(current_oauth_token)

    log.debug("Creating secure gRPC channel")
    channel_credentials = grpc.ssl_channel_credentials()
    if call_credentials is not None:
        channel_credentials = grpc.composite_channel_credentials(channel_credentials, call_credentials)
    channel = grpc.secure_channel(args.serverUrl, credentials=channel_credentials)

    return channel

def create_get_wordset_metadata_request(artifactUrn):
    artifact_reference = ResourceReference(uri=artifactUrn)
    return GetWordsetMetadataRequest(artifact_reference=artifact_reference)

def create_delete_wordset_request(artifactUrn):
    artifact_reference = ResourceReference(uri=artifactUrn)
    return DeleteWordsetRequest(artifact_reference=artifact_reference)

def list_to_dict(list, separator = ':'):
  if not list: 
     return {}
  else:
     return dict(entry.split(separator, 1) for entry in list)

def create_compile_wordset_request(args):
  target_artifact_reference = ResourceReference(uri=args.artifactUrn)
  companion_artifact_reference = ResourceReference(uri=args.modelUrn)
  wordset = json.dumps(json.load(args.wordsetFile))
  return CompileWordsetRequest(
    target_artifact_reference=target_artifact_reference,
    companion_artifact_reference=companion_artifact_reference,
    wordset=wordset,
    metadata=list_to_dict(args.metadata),
    client_data=list_to_dict(args.clientData))

def compile_wordset(args):
  with create_channel(args) as channel:
    stub = WordsetStub(channel)
    compiled_wordset_request = create_compile_wordset_request(args)
    for message in stub.CompileWordsetAndWatch(compiled_wordset_request):
      print(MessageToJson(message))

def get_wordset_metadata(args):
  with create_channel(args) as channel:
    stub = WordsetStub(channel)
    response = stub.GetWordsetMetadata(create_get_wordset_metadata_request(args.artifactUrn))
    print(MessageToJson(response))

def delete_wordset(args):
  with create_channel(args) as channel:
    stub = WordsetStub(channel)
    response = stub.DeleteWordset(create_delete_wordset_request(args.artifactUrn))
    print(MessageToJson(response))

def main():
  parse_args()
  log_level = logging.DEBUG
  global log
  log = logging.getLogger('')
  logging.basicConfig(
      format='%(asctime)s %(levelname)-5s: %(message)s', level=log_level)
    
  if args.oauthURL:
      if args.clientID is None:
          log.error("OAuth 2.0 URL was supplied but client ID is missing")
          return
      elif args.clientSecret is None:
          log.error("OAuth 2.0 URL was supplied but client secret is missing")
          return

  get_oauth2_token()
  switcher = {
    'compile' : compile_wordset,
    'get-metadata' : get_wordset_metadata,
    'delete' : delete_wordset
  }
  switcher.get(args.command)(args)
  print("Done")

if __name__ == '__main__':
  main()